// --------------------------------------------------------------------
// Loss rate
// --------------------------------------------------------------------
function getLossRate(seq_id) {
    // Check for new test loop
    if (seq_id == 0) {
        // new test loop
        // node.warn("New test loop started");
        global.set('received', 1);
        return 0;
    }
    // Get number of received packets
    var received = global.get('received');
    // node.warn("received = " + received);
    // Calculate loss rate;
    var loss_rate = (seq_id - received) / seq_id;
    // Update received number
    // global.set('received', received + 1);
    // Create integer (times 10)
    var loss_rate_int = Math.ceil(loss_rate * 10);
    // node.warn("Loss rate: " + loss_rate + " Int " + loss_rate_int);
    return loss_rate_int;
}

// --------------------------------------------------------------------
// Distance calculation
// --------------------------------------------------------------------

const EARTH_RADIUS = 6371000;

function degreesToRadians(degrees) {
    return degrees * (Math.PI / 180);
}

function radiansToDegrees(radians) {
    return radians * (180 / Math.PI);
}

function angularDistance(location1, location2) {
    const location1_latitude_radians = degreesToRadians(location1.latitude);
    const location2_latitude_radians = degreesToRadians(location2.latitude);
    return Math.acos(
        Math.sin(location1_latitude_radians) * Math.sin(location2_latitude_radians) +
        Math.cos(location1_latitude_radians) * Math.cos(location2_latitude_radians) *
        Math.cos(degreesToRadians(Math.abs(location1.longitude - location2.longitude)))
    );
}

function circleDistance(location1, location2) {
    return EARTH_RADIUS * angularDistance(location1, location2);
}

function constrain(value, min, max) {
    if (value < min) return min;
    if (value > max) return max;
    return value;
}

// --------------------------------------------------------------------
// Data process
// --------------------------------------------------------------------

function ftdProcess(bytes, port, sequence_id, gateways) {

    // Filter wrong messages by length
    if ((1 == port) && (bytes.length != 10)) return null;

    var len = bytes.length;
    node.warn("Got " + len + " bytes");
    // Get packet loss rate
    var packet_num = bytes[8] << 8;
    packet_num = packet_num + bytes[9];
    var loss_rate = getLossRate(packet_num);

    // output object
    var data = {};

    // Check if location is present
    data.has_gps = false;
    for (var idx = 0; idx < 6; idx++) {
        if (bytes[idx] != 0) {
            data.has_gps = true;
        }
    }

    if (data.has_gps) {
        // decode bytes    
        var lonSign = (bytes[0] >> 7) & 0x01 ? -1 : 1;
        var latSign = (bytes[0] >> 6) & 0x01 ? -1 : 1;
        var encLat = ((bytes[0] & 0x3f) << 17) + (bytes[1] << 9) + (bytes[2] << 1) + (bytes[3] >> 7);
        var encLon = ((bytes[3] & 0x7f) << 16) + (bytes[4] << 8) + bytes[5];

        // get tester location
        data.latitude = latSign * (encLat * 108 + 53) / 10000000;
        data.longitude = lonSign * (encLon * 215 + 107) / 10000000;
        node.warn("Tester Lat " + data.latitude + " Lng " + data.longitude);
    }
    else {
        data.latitude = 0.0;
        data.longitude = 0.0;
    }

    // build gateway data
    data.num_gateways = gateways.length;
    data.min_distance = 0;
    data.max_distance = 0;
    data.min_rssi = 200;
    data.max_rssi = -200;
    data.min_snr = 200;
    data.max_snr = -200;
    data.gw_id_1 = 0;
    data.gw_id_2 = 0;
    data.gw_id_3 = 0;
    data.got_gw_id = false;
    gateways.forEach(function (gateway) {
        if (gateway.rssi < data.min_rssi) data.min_rssi = gateway.rssi;
        if (gateway.rssi > data.max_rssi) data.max_rssi = gateway.rssi;
        if (gateway.snr < data.min_snr) data.min_snr = gateway.snr;
        if (gateway.snr > data.max_snr) data.max_snr = gateway.snr;
        if ((data.has_gps) && (gateway.location)) {
            var distance = parseInt(circleDistance(data, gateway.location));
            node.warn("Calc distance " + distance);
            if (distance < data.min_distance) data.min_distance = distance;
            if (distance > data.max_distance) data.max_distance = distance;
        }
        node.warn("Gateway lat " + gateway.location.latitude + " lng " + gateway.location.longitude);

        // node.warn("GW ID = " + gateway.gatewayId);
        if (data.got_gw_id == false) {
            // As we are not on a gateway, use first gateways EUI
            data.got_gw_id = true;
            node.warn("GW ID = " + gateway.gatewayId);
            var gw_id_num = gateway.gatewayId[10] + gateway.gatewayId[11];
            data.gw_id_1 = parseInt(gw_id_num, 16);
            // node.warn("GW ID 1 = " + data.gw_id_1);
            gw_id_num = gateway.gatewayId[12] + gateway.gatewayId[13];
            data.gw_id_2 = parseInt(gw_id_num, 16);
            // node.warn("GW ID 2 = " + data.gw_id_2);
            gw_id_num = gateway.gatewayId[14] + gateway.gatewayId[15];
            data.gw_id_3 = parseInt(gw_id_num, 16);
            // node.warn("GW ID 2 = " + data.gw_id_3);
        }
    });

    node.warn("used GW ID = ac1f09fffe0" + data.gw_id_1.toString(16) + data.gw_id_2.toString(16) + data.gw_id_3.toString(16));
    // data.gw_id_1 = data.gw_id_2 = data.gw_id_3 = 0xff;
    // build response buffer
    var min_distance = data.has_gps ? constrain(Math.round(data.min_distance / 250.0), 1, 128) : 0;
    var max_distance = data.has_gps ? constrain(Math.round(data.max_distance / 250.0), 1, 128) : 0;
    node.warn("Min Distance: " + data.min_distance + " Max Distance: " + data.max_distance);
    node.warn("Max SNR " + data.max_snr)
    var send_snr = parseInt(data.max_snr + 200, 10) & 0xFF;
    // node.warn("Send Max SNR " + send_snr)
    data.buffer = Buffer.from([
        (loss_rate >> 8) & 0x00FF,
        loss_rate & 0xFF,
        parseInt(data.max_rssi + 200, 10) & 0xFF,
        min_distance,
        max_distance,
        ((data.num_gateways << 4) & 0xFF) + (sequence_id >> 4), // create new GW number << 4 + sequenceID >> 4
        sequence_id & 0x0F,
        parseInt(data.max_snr, 10) & 0xFF,
        data.gw_id_1, data.gw_id_2, data.gw_id_3
    ])

    return data;
}

function parser_cs34(msg) {

    if (typeof (msg.payload.fPort) == "undefined") {
        // node.warn("No fPort ");
        return null;
    }
    // avoid sending downlink ACK to integration
    var port = msg.payload.fPort;
    node.warn("fPort " + port);
    if ((port != 1) && (port != 11) && (port != 5)) {
        return null;
    }
    // get version
    var version = ('deviceInfo' in msg.payload) ? 4 : 3;
    node.warn("version " + version);
    // get raw input
    var bytes = Buffer.from(msg.payload.data, 'base64');
    var sequence_id = msg.payload.fCnt;
    var gateways = msg.payload.rxInfo;
    var dev_eui = (version == 4) ? msg.payload.deviceInfo.devEui : msg.payload.devEui;
    node.warn("DevEUI " + dev_eui);

    // get response
    var data = ftdProcess(bytes, port, sequence_id, gateways);
    if (!data) return null;

    // build response for Chirpstack
    msg.data = data;
    msg.topic = msg.topic.replace('/event/up', '/command/down');
    msg.payload = {
        "confirmed": false,
        "fPort": 3,
        "data": data.buffer.toString('base64')
    }
    if (version == 4) {
        msg.payload.devEui = dev_eui;
    }

    return msg;

}

msg = parser_cs34(msg);


return msg;