#include <string.h>
#include <zephyr/device.h>
#include <zephyr/kernel.h>
#include <zephyr/net/socket.h>
#include <zephyr/net/net_if.h>
#include <zephyr/net/dns_resolve.h>
#include <zephyr/pm/device.h>
#include <zephyr/pm/device_runtime.h>

#include <zephyr/drivers/cellular.h>

#include <zephyr/logging/log.h>
LOG_MODULE_REGISTER(main, LOG_LEVEL_DBG);

const struct device *modem = DEVICE_DT_GET(DT_ALIAS(modem));

static const char* cellular_registration_status_to_string(enum cellular_registration_status status) {
    switch (status) 
	{
        case CELLULAR_REGISTRATION_NOT_REGISTERED:
            return "Not Registered";
        case CELLULAR_REGISTRATION_REGISTERED_HOME:
            return "Registered Home";
        case CELLULAR_REGISTRATION_SEARCHING:
            return "Searching";
        case CELLULAR_REGISTRATION_DENIED:
            return "Denied";
        case CELLULAR_REGISTRATION_UNKNOWN:
            return "Unknown";
        case CELLULAR_REGISTRATION_REGISTERED_ROAMING:
            return "Registered Roaming";
        default:
            return "Invalid Status";
    }
}

static void print_cellular_info(void)
{
	int rc;
	char buffer[64];

    LOG_INF("================================");

	// Modem info
	rc = cellular_get_modem_info(modem, CELLULAR_MODEM_INFO_IMEI, &buffer[0], sizeof(buffer));
	if (!rc) {
		LOG_INF("IMEI: %s", buffer);
	}
	rc = cellular_get_modem_info(modem, CELLULAR_MODEM_INFO_MODEL_ID, &buffer[0],
				     sizeof(buffer));
	if (!rc) {
		LOG_INF("MODEL_ID: %s", buffer);
	}
	rc = cellular_get_modem_info(modem, CELLULAR_MODEM_INFO_MANUFACTURER, &buffer[0],
				     sizeof(buffer));
	if (!rc) {
		LOG_INF("MANUFACTURER: %s", buffer);
	}
	rc = cellular_get_modem_info(modem, CELLULAR_MODEM_INFO_FW_VERSION, &buffer[0],
				     sizeof(buffer));
	if (!rc) {
		LOG_INF("FW_VERSION: %s", buffer);
	}
	rc = cellular_get_modem_info(modem, CELLULAR_MODEM_INFO_SIM_IMSI, &buffer[0],
				     sizeof(buffer));
	if (!rc) {
		LOG_INF("SIM_IMSI: %s", buffer);
	}
	rc = cellular_get_modem_info(modem, CELLULAR_MODEM_INFO_SIM_ICCID, &buffer[0],
				     sizeof(buffer));
	if (!rc) {
		LOG_INF("SIM_ICCID: %s", buffer);
	}

	k_sleep(K_MSEC(5000));
	int16_t rssi = 0;
	rc = cellular_get_signal(modem, CELLULAR_SIGNAL_RSSI, &rssi);
	if (!rc) {
		LOG_INF("RSSI %d", rssi);
	} else {
		LOG_ERR("RSSI get error: %d", rc);
	}

	// Registration Status
	enum cellular_access_technology tech = CELLULAR_ACCESS_TECHNOLOGY_LTE_CAT_M1;
    enum cellular_registration_status status = CELLULAR_REGISTRATION_NOT_REGISTERED;
    rc = cellular_get_registration_status(modem, tech, &status);
    if (rc) {
        LOG_ERR("Failed to get registration status: %d", rc);
    } else {
		LOG_INF("Tech: %s, Status: %s", "LTE-M", cellular_registration_status_to_string(status));
	}

	tech = CELLULAR_ACCESS_TECHNOLOGY_EDGE;
    status = CELLULAR_REGISTRATION_NOT_REGISTERED;
    rc = cellular_get_registration_status(modem, tech, &status);
    if (rc) {
        LOG_ERR("Failed to get registration status: %d", rc);
    } else {
		LOG_INF("Tech: %s, Status: %s", "EDGE", cellular_registration_status_to_string(status));
	}

	LOG_INF("================================");
}

int main(void)
{
	struct net_if *const iface = net_if_get_first_by_type(&NET_L2_GET_NAME(PPP));
	int ret;

	LOG_INF("Powering on modem");
	pm_device_action_run(modem, PM_DEVICE_ACTION_RESUME);

	LOG_INF("Bring up network interface");
	ret = net_if_up(iface);
	if (ret < 0) {
		LOG_ERR("Failed to bring up network interface");
		return -1;
	}

	LOG_INF("Waiting for L4 connected");
	ret = net_mgmt_event_wait_on_iface(iface, NET_EVENT_L4_CONNECTED, NULL, NULL, NULL,
					   K_SECONDS(120)); // Waiting 2 minutes there!!!

	if (ret != 0) {
		LOG_ERR("L4 was not connected in time");
		return -1;
	}

	LOG_INF("Waiting for DNS server added");
	ret = net_mgmt_event_wait_on_iface(iface, NET_EVENT_DNS_SERVER_ADD, NULL, NULL, NULL,
					   K_SECONDS(10));
	if (ret) {
		LOG_ERR("DNS server was not added in time");
		return -1;
	}

	print_cellular_info();

	ret = net_if_down(iface);
	if (ret < 0) {
		LOG_ERR("Failed to bring down network interface");
		return -1;
	}

	LOG_INF("Powering down modem");
	ret = pm_device_action_run(modem, PM_DEVICE_ACTION_SUSPEND);
	if (ret != 0) {
		LOG_ERR("Failed to power down modem");
		return -1;
	}

	LOG_INF("Sample complete");

	while (1) {
		k_sleep(K_FOREVER);
	}

	return 0;
}
