/**
 * @file RUI3-Modular.ino
 * @author Bernd Giesecke (bernd@giesecke.tk)
 * @brief RUI3 based code for low power practice
 * @version 0.1
 * @date 2023-03-29
 *
 * @copyright Copyright (c) 2023
 *
 */
#include "app.h"

// Set sender or receiver mode
#define IS_RECEIVER 1
// #define IS_SENDER 1

/** Flag if transmit is active, used by some sensors */
volatile bool tx_active = false;

/** Payload buffer */
uint8_t g_solution_data[64];

uint16_t my_fcount = 1;

volatile bool got_ack = false;

/**
 * @brief LoRa P2P callback if a packet was received
 *
 * @param data pointer to the data with the received data
 */
void recv_cb(rui_lora_p2p_recv_t data)
{
	// MYLOG("RX-P2P-CB", "P2P RX, RSSI %d, SNR %d", data.Rssi, data.Snr);
	// for (int i = 0; i < data.BufferSize; i++)
	// {
	// 	Serial.printf("%02X", data.Buffer[i]);
	// }
	// Serial.print("\r\n");
	tx_active = false;
#ifdef IS_RECEIVER
	MYLOG("RX-P2P-CB", "Prepare to send ACK");
	// Send an ACK
	api.system.timer.start(RAK_TIMER_1, 1000, NULL);
#else
	MYLOG("RX-P2P-CB", "ACK received, sleep");
	got_ack = true;
	// Start timer to disable RX
	api.system.timer.stop(RAK_TIMER_1);
	api.system.timer.start(RAK_TIMER_1, 1000, NULL);
#endif
}

/**
 * @brief LoRa P2P callback if a packet was sent
 *
 */
void send_cb(void)
{
	// MYLOG("TX-P2P-CB", "P2P TX finished");
	digitalWrite(LED_BLUE, LOW);
	tx_active = false;
#ifdef IS_RECEIVER
	MYLOG("TX-P2P-CB", "ACK sent");
#else
	MYLOG("TX-P2P-CB", "Packet sent out");
#endif
}

/**
 * @brief LoRa P2P callback for CAD result
 *
 * @param result true if activity was detected, false if no activity was detected
 */
void cad_cb(bool result)
{
	MYLOG("CAD-P2P-CB", "P2P CAD reports %s", result ? "activity" : "no activity");
}

/**
 * @brief Arduino setup, called once after reboot/power-up
 *
 */
void setup()
{
	// Setup for LoRaWAN
	if (api.lorawan.nwm.get() == 1)
	{
		api.lora.nwm.set();
	}

	api.lora.registerPRecvCallback(recv_cb);
	api.lora.registerPSendCallback(send_cb);
	api.lora.registerPSendCADCallback(cad_cb);

	// Enable RX with TX enabled
	api.lora.precv(65533);

	pinMode(LED_GREEN, OUTPUT);
	digitalWrite(LED_GREEN, HIGH);
	pinMode(LED_BLUE, OUTPUT);
	digitalWrite(LED_BLUE, HIGH);

	pinMode(WB_IO2, OUTPUT);
	digitalWrite(WB_IO2, LOW);

	// Start Serial
	Serial.begin(115200);

	// Delay for 5 seconds to give the chance for AT+BOOT
	delay(5000);

#ifdef IS_RECEIVER
	Serial.println("RAKwireless RUI3 P2P Receiver Test");
	api.system.firmwareVersion.set("RUI3-P2P_Receiver");
#else
	Serial.println("RAKwireless RUI3 P2P Sender Test");
	api.system.firmwareVersion.set("RUI3-P2P_Sender");
#endif
	Serial.println("------------------------------------------------------");
	Serial.println("Setup the device with WisToolBox or AT commands before using it");
	Serial.printf("Version %s\n", api.system.firmwareVersion.get().c_str());
	Serial.printf("Stack %s\n", api.lorawan.ver.get().c_str());
	Serial.println("------------------------------------------------------");

	// Initialize module
	Wire.begin();

	// Register the custom AT command to get device status
	if (!init_status_at())
	{
		MYLOG("SETUP", "Add custom AT command STATUS fail");
	}

	// Register the custom AT command to set the send interval
	if (!init_interval_at())
	{
		MYLOG("SETUP", "Add custom AT command Send Interval fail");
	}

	// Get saved sending interval from flash
	get_at_setting();

	digitalWrite(LED_GREEN, LOW);

	// Create a timer.
	api.system.timer.create(RAK_TIMER_0, sensor_handler, RAK_TIMER_PERIODIC);
#ifdef IS_RECEIVER
	// Send interval forced to off, no timer for automatic sending
	custom_parameters.send_interval = 0;
#else
	if (custom_parameters.send_interval == 0)
	{
		custom_parameters.send_interval = 15000;
	}
	if (custom_parameters.send_interval != 0)
	{
		// Start a timer.
		api.system.timer.start(RAK_TIMER_0, custom_parameters.send_interval, NULL);
	}

#endif

#ifdef IS_RECEIVER
	// Create timer to send ACK
	api.system.timer.create(RAK_TIMER_1, send_ack, RAK_TIMER_ONESHOT);
#else
	// Create timer for timeout of ACK
	api.system.timer.create(RAK_TIMER_1, resend_packet, RAK_TIMER_ONESHOT);
#endif

	// Enable low power mode
#if defined(_VARIANT_RAK3172_) || defined(_VARIANT_RAK3172_SIP_)
	api.system.lpmlvl.set(2);
#endif
	// api.system.lpm.set(1);
	api.system.lpm.set(0);

	// If available, enable BLE advertising for 30 seconds and open the BLE UART channel
#if defined(_VARIANT_RAK3172_) || defined(_VARIANT_RAK3172_SIP_)
// No BLE
#else
	// Serial6.begin(115200, RAK_AT_MODE);
	// api.ble.advertise.start(30);
#endif
}

/**
 * @brief sensor_handler is a timer function called every
 * custom_parameters.send_interval milliseconds. Default is 120000. Can be
 * changed with ATC+SENDINT command
 *
 */
void sensor_handler(void *)
{
	// MYLOG("UPLINK", "Start");
	digitalWrite(LED_BLUE, HIGH);

	// Check if last TX cycle is finished
	if (tx_active)
	{
		MYLOG("UPLINK", "TX still ongoing, skip sending");
		return;
	}

	// Create payload (Cayenne LPP format for voltage)
	g_solution_data[0] = 0x01;
	g_solution_data[1] = 0x74;
	g_solution_data[2] = 0x01;
	g_solution_data[3] = 0x8c;

	// Send the packet
	send_packet();
}

/**
 * @brief Send the data packet that was prepared in
 * Cayenne LPP format by the different sensor and location
 * aqcuision functions
 *
 */
void send_packet(void)
{
	MYLOG("SEND", "Send packet with size 4 over P2P");
	digitalWrite(LED_BLUE, LOW);

#ifdef IS_RECEIVER
	// Switch off RX mode
	// api.lora.precv(65533);
#else
	// Enable RX for the ACK with TX enabled
	api.lora.precv(65533);
	MYLOG("SEND", "RX enabled");
#endif

	if (api.lora.psend(4, g_solution_data, false))
	{
		// MYLOG("UPLINK", "Packet enqueued");
		api.system.timer.stop(RAK_TIMER_1);
		api.system.timer.start(RAK_TIMER_1, 15000, NULL);
	}
}

void resend_packet(void *)
{
	if (!got_ack)
{	MYLOG("RESEND", "Resend packet with size 4 over P2P");

	digitalWrite(LED_BLUE, LOW);
	// We send second packet, disable RX
	api.lora.precv(0);
	MYLOG("RESEND", "RX disabled");
	delay(500);
	if (api.lora.psend(4, g_solution_data, false))
	{
		// MYLOG("UPLINK", "Packet enqueued");
	}
}
else
{
	got_ack = false;
	// Switch off RX
	api.lora.precv(0);
	MYLOG("ACK_RCVD", "RX disabled");
}
}

void send_ack(void *)
{
	MYLOG("ACK", "Send ACK packet");

	digitalWrite(LED_BLUE, LOW);

	if (api.lora.psend(1, g_solution_data, false))
	{
		// MYLOG("UPLINK", "Packet enqueued");
	}
	else
	{
		MYLOG("ACK", "Send failed");
	}
}

/**
 * @brief This example is complete timer driven.
 * The loop() does nothing than sleep.
 *
 */
void loop()
{
	// api.system.sleep.all();
	api.system.scheduler.task.destroy();
}
